import * as d3 from 'd3';

const createBarChart = (eventBus) => {
    const svgHeight = 500;
    const svgWidth = 1000;
    const paddingLeft = 70;
    const paddingBottom = 130;

    const svg = d3.select("body")
        .append("svg")
        .attr("width", svgWidth)
        .attr("height", svgHeight);

    const barsG = svg.append("g")
        .attr("transform", `translate(${paddingLeft}, 0)`);

    const xAxisG = svg.append("g")
        .attr("transform", `translate(${paddingLeft}, ${svgHeight - paddingBottom})`);

    const yAxisG = svg.append("g")
        .attr("transform", `translate(${paddingLeft}, 0)`);

    //relevant for TASK: interactivity
    const tooltipDiv = d3.select("body")
        .append("div")
        .style("position", "fixed")
        .style("width", "75px")
        .style("height", "18px")
        .style("top", "0px")
        .style("left", "0px")
        .style("z-index", 1)
        .style("text-align", "center")
        .style("background-color", "yellow")
        .style("visibility", "hidden");

    let selectedContinent = null;

    eventBus.continentClicked = async (continentLabel, color) => {
        let data;

        if (selectedContinent === continentLabel) {
            data = [];
            selectedContinent = null;
        } else {
            const continentLowerCase = continentLabel.toLowerCase().replace(" ", "_");

            try {
                if (continentLowerCase === "north_america") {
                    /* TASK: Data requesting and parsing */
                    /* replace data = [] */
                    data = []
                } else {
                    data = await d3.json(`./data/population_${continentLowerCase}.json`);
                }
                selectedContinent = continentLabel;
            } catch(e) {
                console.log("Sorry. No data available. Please test with Europe and South America.");
                data = [];
				selectedContinent = null;
            }
        }

        data.forEach(d => {
            d.population = Number(d.population);
        });

        const maxPopulation = d3.max(data, d => d.population);

        const countries = data.map(d => d.countryLabel);

        //padding value has to be between 0 and 1
        const scaleX = d3.scaleBand()
            .domain(countries)
            .rangeRound([0, svgWidth - paddingLeft])
            .padding(0.05);

        //coordinate y-axis is inverted
        const scaleY = d3.scaleLinear()
            .domain([0, maxPopulation])
            .range([svgHeight - paddingBottom, 0]);

        const selectedBars = barsG.selectAll("rect")
            .data(data);

        /* TASK: Data binding */
        /* add method after .append("rect") */
        const bars = selectedBars.enter()
            .append("rect");

        /* TASK: Animation */
        /* modify y and height */
        /* (see second part below) */
        bars.attr("x", d => scaleX(d.countryLabel))
            .attr("y", d => scaleY(d.population))
			.attr("width", scaleX.bandwidth())
            .attr("height", d => svgHeight - paddingBottom - scaleY(d.population))
            .attr("class", "bar")
            .style("fill", color)
            .on("click", (d) => {
                const countryColors = {
                    "Switzerland": "red",
                    "Germany": "green"
                };

                eventBus.countryClicked(d.countryLabel, countryColors[d.countryLabel]);
            });
        /* TASK: Interactivity */
        /* enter code here */

        /* TASK: Animation */
        /* enter code here */
        /* (see first part above) */

        selectedBars.exit().remove();

        const xAxis = d3.axisBottom(scaleX);

        xAxisG.call(xAxis);

        /* TASK: Selections */
        /* enter code here */

        //rotates the labels of the x-axis
        xAxisG.selectAll(".tick text")
            .style("dominant-baseline", "middle") //set rotation center of label to the middle
            .attr("y", 0)
            .attr("dy", 0)
            .style("transform", (d, i, e) => {
                const bbox = e[i].getBBox();
                const labelWidthHalf = bbox.width / 2 + 4;
                //note that transformations are applied in reverse order
                return `translate(0, ${labelWidthHalf}px)rotate(-90deg)`
            });

        const yAxis = d3.axisLeft(scaleY);

        yAxisG.call(yAxis);
    };
};

export default createBarChart;