import * as d3 from "d3";

const log = console.log;

const helloWorld = () => {
    d3.select("body")
        .append("p")
        .text("Hello World!");
};

const methodChaining = () => {
    const svg = d3.select("body")
        .append("svg")
        .attr("width", 400)
        .attr("height", 225)
        .style("background-color", "green");

    const svgWidth = svg.attr("width");
    log(svgWidth);
};

const selections = () => {
    document.body.innerHTML = `
    <div id="app">
        <div class="container">
            <svg>
                <rect x="10" y="10" width="280" height="130" fill="red"></rect>
            </svg>
        </div>
        <div class="container">
            <svg>
                <rect x="10" y="10" width="280" height="130" fill="blue"></rect>
            </svg>
        </div>        
    </div>
    `;

    const appElement = d3.select("#app");
    log(appElement.node());
    const containerElements = appElement.selectAll(".container");
    log(containerElements.nodes());
    const image2Element = containerElements.select("rect[fill='blue']");
    log(image2Element.node());
};

const bind = data => {
    const divs = d3.select("body")
        .selectAll("div")
        .data(data)
        .text((d, i) => "Updated Div " + i)
        .style("color", d => d);

    divs.enter()
        .append("div")
        .text((d, i) => "New Div " + i)
        .style("color", d => d);

    divs.exit()
        .text((d, i) => "Removed Div " + i)
        .style("color", "black");
    //or use .remove() to delete elements from the DOM
    log(divs.nodes());
};

const dataBinding = () => {
    bind(["red", "green", "blue"]);
    //bind(["green", "blue", "red"]);
    //bind([]);
};

const dataRequestingAndParsing = async () => {
    const data = await d3.csv("data/cars.csv", d => ({
        year: new Date(Number(d.Year), 0, 1),
        make: d.Make,
        model: d.Model,
        length: Number(d.Length)
    }));

    log(data[0]);
    log(data[1]);
};

const scales = () => {
    const t = d3.scaleThreshold()
        .domain([1, 3])
        .range(["a", "b", "c"]);

    log(t(0));
    log(t(1));
    log(t(2));
    log(t(3));
    log(t(1000));
};

const axes = () => {
    const data = [
        {"key": "a", "value": 300},
        {"key": "b", "value": 100},
        {"key": "c", "value": 200}
    ];

    const xScale = d3.scaleOrdinal()
        .domain(["a", "b", "c"])
        .range([0, 50, 100]);

    const yScale = d3.scaleLinear()
        .domain([0, 500])
        .range([0, 200]);

    const svgG = d3.select("body")
        .append("svg")
        .append("g")
        .attr("transform", "translate(30, 20)");

    svgG.selectAll("circle")
        .data(data)
        .enter()
        .append("circle")
        .attr("cx", d => xScale(d.key))
        .attr("cy", d => yScale(d.value))
        .attr("r", 3);

    const xAxis = d3.axisTop(xScale);
    const yAxis = d3.axisLeft(yScale);

    svgG.append("g")
        .call(xAxis);

    svgG.append("g")
        .call(yAxis);
};

const animation = () => {
    const svg = d3.select("body")
        .append("svg");

    const circle = svg.append("circle")
        .attr("cx", 10)
        .attr("cy", 10)
        .attr("r", 3);

    const xTransition = circle
        .transition()
        .delay(1000)
        .duration(3000)
        .attr("cx", 100);

    xTransition
        .transition()
        .ease(d3.easeElastic)
        .duration(3000)
        .attr("cy", 100);
};

const interactivity = () => {
    let backgroundColor = "";

    const svg = d3.select("body")
        .append("svg");

    svg.on("click", () => {
        if (backgroundColor === "green")
            backgroundColor = "red";
        else
            backgroundColor = "green";

        svg.style("background-color", backgroundColor)
    });

    svg.dispatch("click");
};

const generatorFunctions = () => {
    const pieGenerator = d3.pie();
    const pieData = pieGenerator([1, 2, 3, 4]);
    console.log(pieData);
};

const helperFunctions = () => {
    const extent = d3.extent([3, -1, 2, 4]);
    console.log(extent);

    const numberFormatter = d3.format(",");
    const formattedNumber = numberFormatter(1000000);
    console.log(formattedNumber);

    const color = d3.rgb("limegreen");
    console.log(color);
    console.log(color.hex());
};

/* uncomment a method to run a specific snippet */
const runSnippets = () => {
    //helloWorld();
    //methodChaining();
    //selections();
    //dataBinding();
    //dataRequestingAndParsing();
    //scales();
    //axes();
    //animation();
    //interactivity();
    //generatorFunctions();
    //helperFunctions();
};

export default runSnippets;